// Generic Service Worker (sw.js) - Proxy Approach
// This file should be placed at the root of the client's website.

self.addEventListener('push', function(event) {
    console.log('[Service Worker] Push Received.');

    let notificationData = {};
    try {
        // Attempt to parse the push data as JSON
        // Expecting: { "notificationInstanceId": "some-unique-id" }
        const pushPayload = event.data ? event.data.json() : {};
        const notificationInstanceId = pushPayload.notificationInstanceId;

        if (!notificationInstanceId) {
            console.error('[Service Worker] Push payload does not contain notificationInstanceId.');
            // Optionally show a default notification if payload is missing ID
            // return; 
            // For now, we'll try to show a default notification later if fetch fails
        }

        console.log(`[Service Worker] Fetching notification details for ID: ${notificationInstanceId}`);

        // Fetch the actual notification details from the server API
        // IMPORTANT: Replace 'https://YOUR_SERVER_URL.com' with the actual server URL
        const serverUrl = 'https://notificacaoteste-notificacaoteste2.xb7m7k.easypanel.host'; 
        const apiUrl = `${serverUrl}/api/notification-details/${notificationInstanceId}`;

        const fetchPromise = fetch(apiUrl)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`Server responded with status: ${response.status}`);
                }
                return response.json();
            })
            .then(details => {
                console.log('[Service Worker] Notification details fetched:', details);
                // Validate fetched details (ensure required fields exist)
                if (!details.title || !details.body || !details.data || !details.data.url) {
                    throw new Error('Fetched notification details are incomplete.');
                }
                // Prepare notification options based on fetched details
                notificationData = {
                    title: details.title,
                    options: {
                        body: details.body,
                        icon: details.icon || '/img/default-logo.png', // Use default icon if not provided
                        badge: details.badge || '/img/badge.png', // Optional: small icon
                        data: {
                            url: details.data.url // URL to open on click
                        }
                        // Add other options like image, actions, etc. if needed
                    }
                };
                return self.registration.showNotification(notificationData.title, notificationData.options);
            })
            .catch(error => {
                console.error('[Service Worker] Failed to fetch notification details or show notification:', error);
                // Fallback: Show a generic notification if fetching details fails
                const fallbackTitle = 'Nova Notificação';
                const fallbackOptions = {
                    body: 'Clique para ver as novidades.',
                    icon: '/img/default-logo.png',
                    badge: '/img/badge.png',
                    data: {
                        url: '/' // Default URL
                    }
                };
                return self.registration.showNotification(fallbackTitle, fallbackOptions);
            });

        event.waitUntil(fetchPromise);

    } catch (e) {
        console.error('[Service Worker] Error processing push event:', e);
        // Fallback for JSON parsing errors or other unexpected issues
        const fallbackTitle = 'Erro na Notificação';
        const fallbackOptions = {
            body: 'Ocorreu um erro ao processar a notificação.',
            icon: '/img/default-logo.png',
            badge: '/img/badge.png',
            data: {
                url: '/' 
            }
        };
        event.waitUntil(self.registration.showNotification(fallbackTitle, fallbackOptions));
    }
});

self.addEventListener('notificationclick', function(event) {
    console.log('[Service Worker] Notification click Received.');

    event.notification.close(); // Close the notification

    // Open the URL specified in the notification's data
    const urlToOpen = event.notification.data.url;
    if (urlToOpen) {
        event.waitUntil(
            clients.matchAll({ type: 'window', includeUncontrolled: true }).then(windowClients => {
                // Check if a window/tab with the same URL is already open
                for (var i = 0; i < windowClients.length; i++) {
                    var client = windowClients[i];
                    if (client.url === urlToOpen && 'focus' in client) {
                        return client.focus();
                    }
                }
                // If no window/tab is open, open a new one
                if (clients.openWindow) {
                    return clients.openWindow(urlToOpen);
                }
            })
        );
    } else {
        console.warn('[Service Worker] No URL found in notification data to open.');
    }
});

// Optional: Add listeners for activate, install if needed for caching, etc.
self.addEventListener('install', event => {
  console.log('[Service Worker] Installing...');
  // Skip waiting forces the waiting service worker to become the active service worker.
  self.skipWaiting();
});

self.addEventListener('activate', event => {
  console.log('[Service Worker] Activating...');
  // Claiming clients allows the service worker to control pages immediately.
  event.waitUntil(clients.claim());
});

